/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
head_score();

/**
 * Performs the head score calculation for the specified models
 */
function head_score() {
    let template = Template.GetCurrent();
    let models = ["M1"];
    let body_region_label = "head";
    // Duke Xiao 18-Jul-2025 - Combine WC 1-6 7-8 in CNCAP_2024_Far_Side_VTC_Single_Occupant_1-8_CN.ort
    if (template.filename.includes("CNCAP_2024_Far_Side_VTC_Single_Occupant")) {
        models = ["CASE_7", "CASE_8"];
    }
    let occupants = ["DRIVER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Get the head excursion zone and whether there's a countermeasure or not - required to set the high score value */
            let head_excursion_zone = get_variable_value(status, `${m}_${occ}_DRIVER_HEAD_EXCURSION_RATING`, "string");
            get_variable_value(status, `${m}_${occ}_DRIVER_HEAD_EXCURSION_MAX_VALUE`, "string", true);

            new Variable(
                template,
                "SUMMARY_HEAD_OFFSET",
                "Zone used in summary template",
                head_excursion_zone,
                "General",
                false,
                true
            );

            /** Set capping limit variable based on head excursion zone */
            new Variable(
                template,
                `${m}_HEAD_HEZ_CAPPING_LIMIT`,
                `Capping limit for Head Excursion Zone`,
                /capping/i.test(head_excursion_zone) ? "TRUE" : "FALSE",
                "General",
                false,
                true
            );

            let countermeasure_int = get_variable_value(status, `COUNTERMEASURE`, "string");
            if (countermeasure_int === "Yes") {
                countermeasure_int = 1;
            } else {
                countermeasure_int = 0;
            }
            /* comment why checking for countermeasure_int === 1 */
            let countermeasure = countermeasure_int === 1;

            /* TODO - check status object */

            let max_head_score = get_max_head_score(head_excursion_zone, countermeasure);

            new Variable(
                template,
                `${m}_${occ}_HEAD_HI_SCORE`,
                `Max head score for head excursion in ${head_excursion_zone} zone with${
                    countermeasure ? "" : "out"
                } countermeasure.`,
                max_head_score.toString(),
                "General",
                false,
                true
            );

            /* Scale scores from T/HIS to the allowable hi score */
            let HIC_HI_LIMIT = 500.0;
            let HIC_LO_LIMIT = 700.0;
            let HEAD_LO_SCORE = 0;
            let hic = get_variable_value(status, `${m}_${occ}_HEAD_HIC_HIC_VALUE`, "float");
            if (hic === "Missing" || hic === null) {
                var hic_score = 0;
            } else {
                var hic_score = sliding_scale(hic, HIC_HI_LIMIT, HIC_LO_LIMIT, max_head_score, HEAD_LO_SCORE);
            }
            new Variable(
                template,
                `${m}_${occ}_HEAD_HIC_SCORE`,
                `Result from Automotive Assessments Workflow`,
                hic_score.toFixed(3),
                "General",
                false,
                true
            );

            /** Set capping limit variable for HIC */
            new Variable(
                template,
                `${m}_HEAD_HIC_CAPPING_LIMIT`,
                `Capping limit for HIC`,
                hic >= HIC_LO_LIMIT ? "TRUE" : "FALSE",
                "General",
                false,
                true
            );

            let TMS_HI_LIMIT = 72.0; // 3 milisecond acceleration
            let TMS_LO_LIMIT = 80.0; // 3 milisecond acceleration
            let tms = get_variable_value(status, `${m}_${occ}_HEAD_THREE_MS_EXCEEDENCE_3MS_VALUE`, "float");
            if (tms === "Missing" || tms === null) {
                var tms_score = 0;
            } else {
                var tms_score = sliding_scale(tms, TMS_HI_LIMIT, TMS_LO_LIMIT, max_head_score, HEAD_LO_SCORE);
            }
            new Variable(
                template,
                `${m}_${occ}_HEAD_THREE_MS_EXCEEDENCE_SCORE`,
                `Result from Automotive Assessments Workflow`,
                tms_score.toFixed(3),
                "General",
                false,
                true
            );

            /** Set capping limit variable for TMS */
            new Variable(
                template,
                `${m}_HEAD_TMS_CAPPING_LIMIT`,
                `Capping limit for TMS`,
                tms >= TMS_LO_LIMIT ? "TRUE" : "FALSE",
                "General",
                false,
                true
            );

            let score = Math.min(hic_score, tms_score);
            new Variable(
                template,
                `${m}_HEAD_FINAL_SCORE`,
                `Result from Automotive Assessments Workflow`,
                score.toFixed(3),
                "General",
                false,
                true
            );
        }
    }
}

/**
 * Returns the maximum head score given the head excursion zone
 * and whether there is a countermeasure or not
 * @param {string} zone Head excursion zone
 * @param {boolean} countermeasure Whether there is a countermeasure or not
 * @returns {number}
 */
function get_max_head_score(zone, countermeasure) {
    if (countermeasure) {
        switch (zone) {
            case "CAPPING":
            case "RED":
                return 0;
            case "RED**":
                return 2;
            case "ORANGE":
                return 3;
            case "YELLOW":
                return 4;
            case "GREEN":
                return 4;
            default:
                LogWarning(`Invalid head excursion zone '${zone}' in get_max_head_score(). Returning 0.`);
                return 0;
        }
    } else {
        switch (zone) {
            case "CAPPING":
            case "RED":
                return 0;
            case "ORANGE":
                return 1;
            case "YELLOW":
                return 2;
            case "GREEN":
                return 4;
            default:
                LogWarning(`Invalid head excursion zone '${zone}' in get_max_head_score(). Returning 0.`);
                return 0;
        }
    }
}

function sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score) {
    var retval = 0.0;

    if (val < hi_perf) retval = hi_score;
    else if (val > lo_perf) retval = lo_score;
    else retval = hi_score + ((val - hi_perf) * (lo_score - hi_score)) / (lo_perf - hi_perf);

    return retval;
}
